// Boringfy - Content Script for Link Aggregators
// Replaces clickbait headlines with boring, factual titles
// Supports: Reddit, Hacker News, Google News, Lobsters

const API_URL = "https://api.boringfy.org/boringfy";
const POLL_INTERVAL = 5000; // 5 seconds
const MAX_POLLS = 12; // Max 1 minute of polling
const CACHE_PREFIX = "boringfy_"; // Session storage key prefix

// Track which URLs we're polling for
const pendingUrls = new Map(); // url -> { pollCount, element }

// Extension enabled state
let isEnabled = true;

// Detect which site we're on
function detectSite() {
  const host = window.location.hostname;
  if (host.includes('reddit.com')) return 'reddit';
  if (host === 'news.ycombinator.com') return 'hackernews';
  if (host === 'news.google.com') return 'google';
  if (host === 'lobste.rs') return 'lobsters';
  return 'unknown';
}

const currentSite = detectSite();
console.log('[Boringfy] Detected site:', currentSite);

// Session storage cache helpers
function getCachedTitle(url) {
  try {
    return sessionStorage.getItem(CACHE_PREFIX + url);
  } catch (e) {
    console.warn('[Boringfy] Cache read error:', e);
    return null;
  }
}

function setCachedTitle(url, boringTitle) {
  try {
    sessionStorage.setItem(CACHE_PREFIX + url, boringTitle);
  } catch (e) {
    console.warn('[Boringfy] Cache write error:', e);
  }
}

// Revert all replaced titles to original
function revertAllTitles() {
  console.log('[Boringfy] Reverting all titles to original');
  const replaced = document.querySelectorAll('.boringfy-replaced');
  replaced.forEach(element => {
    if (element.dataset.originalTitle) {
      element.textContent = element.dataset.originalTitle;
    }
    element.classList.remove('boringfy-replaced');
  });
  const pending = document.querySelectorAll('.boringfy-pending');
  pending.forEach(element => {
    element.classList.remove('boringfy-pending');
  });
  pendingUrls.clear();
}

// Listen for toggle messages from popup
browser.runtime.onMessage.addListener((message) => {
  if (message.type === 'toggle') {
    isEnabled = message.enabled;
    console.log('[Boringfy] Toggle received, enabled:', isEnabled);
    if (isEnabled) {
      boringfy();
    } else {
      revertAllTitles();
    }
  }
});

// Load initial enabled state
browser.storage.local.get('enabled').then((result) => {
  isEnabled = result.enabled !== false;
  console.log('[Boringfy] Initial enabled state:', isEnabled);
  if (isEnabled) {
    boringfy();
  }
}).catch(() => {
  isEnabled = true;
  boringfy();
});

console.log('[Boringfy] Extension loaded on', window.location.href);

// ============================================================
// Site-specific link finders
// ============================================================

// Reddit link finder
function findRedditLinks() {
  const links = new Map();
  const selectors = [
    // New Reddit
    'a[data-click-id="body"][href^="http"]:not([href*="reddit.com"])',
    // Old Reddit  
    'a.title[href^="http"]:not([href*="reddit.com"])',
    // Outbound links
    'a[data-outbound-url]',
    // New Reddit 2024+ - slot based
    'shreddit-post a[slot="full-post-link"][href^="http"]:not([href*="reddit.com"])',
    'article a[href^="http"]:not([href*="reddit.com"]):not([href*="redd.it"])',
  ];

  selectors.forEach(selector => {
    document.querySelectorAll(selector).forEach(link => {
      const url = link.dataset.outboundUrl || link.href;
      if (url && !url.includes('reddit.com') && !url.includes('redd.it') && !links.has(url)) {
        if (isAllowedSource(url)) {
          links.set(url, link);
        }
      }
    });
  });

  return links;
}

// Hacker News link finder
function findHackerNewsLinks() {
  const links = new Map();
  
  // HN structure: <span class="titleline"><a href="...">Title</a>
  // The main story links are in .titleline > a (first child)
  document.querySelectorAll('.titleline > a').forEach(link => {
    const url = link.href;
    // Skip HN internal links (item?id=, from?site=, etc.)
    if (url && !url.includes('ycombinator.com') && !links.has(url)) {
      if (isAllowedSource(url)) {
        links.set(url, link);
      }
    }
  });

  return links;
}

// Strip tracking parameters from URLs for consistent matching
// Google News adds utm_*, gaa_* params that cause URL mismatches
function normalizeUrl(url) {
  try {
    const parsed = new URL(url);
    const paramsToRemove = [];
    for (const key of parsed.searchParams.keys()) {
      if (key.startsWith('utm_') || key.startsWith('gaa_')) {
        paramsToRemove.push(key);
      }
    }
    paramsToRemove.forEach(key => parsed.searchParams.delete(key));
    return parsed.toString();
  } catch (e) {
    return url;
  }
}

// Google News link finder
function findGoogleNewsLinks() {
  const links = new Map();
  
  // Helper to extract URL from jslog attribute
  // jslog contains base64-encoded JSON with the actual URL
  function extractUrlFromJslog(jslog) {
    if (!jslog) return null;
    try {
      // jslog format: "95014; 5:W251bGws..." - the base64 part contains URL
      const match = jslog.match(/5:([A-Za-z0-9+/=]+)/);
      if (match) {
        const decoded = atob(match[1]);
        // Look for URL pattern in decoded string
        const urlMatch = decoded.match(/https?:\/\/[^\s"\\]+/);
        if (urlMatch) {
          // Normalize URL to strip tracking params
          return normalizeUrl(urlMatch[0]);
        }
      }
    } catch (e) {
      // Base64 decode failed, ignore
    }
    return null;
  }

  // Strategy: Find ALL links with ./read/ hrefs and jslog containing base64 URLs
  // This is more stable than relying on obfuscated class names like gPFEn, WwrzSb
  // which Google changes frequently.
  //
  // We look for:
  // 1. Links with href starting with "./read/" (Google News article links)
  // 2. Links with jslog attribute containing "5:" followed by base64 (the encoded URL)
  // 3. For hidden links (aria-hidden), find the visible sibling headline
  
  const processedUrls = new Set();
  
  // Find all links pointing to ./read/ with jslog
  document.querySelectorAll('a[href^="./read/"][jslog*="5:"]').forEach(link => {
    const jslog = link.getAttribute('jslog');
    const realUrl = extractUrlFromJslog(jslog);
    
    console.log('[Boringfy] Google News checking link:', link.className, 'realUrl:', realUrl);
    
    if (!realUrl) {
      console.log('[Boringfy] Google News: no realUrl extracted');
      return;
    }
    if (processedUrls.has(realUrl)) {
      console.log('[Boringfy] Google News: URL already processed');
      return;
    }
    if (!isAllowedSource(realUrl)) {
      console.log('[Boringfy] Google News: URL not in allowed sources');
      return;
    }
    
    // Check if this is a hidden link (aria-hidden="true" or no visible text)
    const isHidden = link.getAttribute('aria-hidden') === 'true' || 
                     link.textContent.trim().length === 0;
    
    console.log('[Boringfy] Google News: isHidden =', isHidden, 'aria-hidden =', link.getAttribute('aria-hidden'), 'text length =', link.textContent.trim().length);
    
    if (isHidden) {
      // Find the visible headline link
      // Google News structure: a.WwrzSb is inside div.XlKvRb, which is inside div.UwIKyb
      // The visible a.gPFEn is a direct child of div.UwIKyb (sibling of div.XlKvRb)
      // So we need to traverse up multiple levels to find it
      
      // Walk up the DOM tree looking for a container with a visible headline
      let ancestor = link.parentElement;
      let found = false;
      
      for (let i = 0; i < 5 && ancestor && !found; i++) {
        const allLinks = Array.from(ancestor.querySelectorAll('a[href^="./read/"]'));
        console.log('[Boringfy] Google News: ancestor level', i, 'class =', ancestor.className, 'found', allLinks.length, 'links');
        
        const visibleLink = allLinks.find(a => a !== link && a.textContent.trim().length > 0);
        
        if (visibleLink) {
          console.log('[Boringfy] Google News found (via ancestor level ' + i + '):', realUrl);
          links.set(realUrl, visibleLink);
          processedUrls.add(realUrl);
          found = true;
          break;
        }
        ancestor = ancestor.parentElement;
      }
      
      if (!found) {
        console.log('[Boringfy] Google News: could not find visible link for', realUrl);
      }
    } else {
      // This link itself has visible text - use it directly
      console.log('[Boringfy] Google News found (direct):', realUrl);
      links.set(realUrl, link);
      processedUrls.add(realUrl);
    }
  });

  // Fallback: look for direct external links in articles
  document.querySelectorAll('article a[href^="http"]:not([href*="google.com"])').forEach(link => {
    const url = link.href;
    if (url && !links.has(url) && isAllowedSource(url)) {
      links.set(url, link);
    }
  });



  return links;
}

// Lobsters link finder
function findLobstersLinks() {
  const links = new Map();
  
  // Lobsters structure: <a class="u-url" href="...">Title</a>
  // Story links have class "u-url" within .story
  document.querySelectorAll('.story .u-url').forEach(link => {
    const url = link.href;
    // Skip Lobsters internal links
    if (url && !url.includes('lobste.rs') && !links.has(url)) {
      if (isAllowedSource(url)) {
        links.set(url, link);
      }
    }
  });

  return links;
}

// Main link finder - dispatches to site-specific finder
function findArticleLinks() {
  console.log('[Boringfy] Finding links for site:', currentSite);
  
  let links;
  switch (currentSite) {
    case 'reddit':
      links = findRedditLinks();
      break;
    case 'hackernews':
      links = findHackerNewsLinks();
      break;
    case 'google':
      links = findGoogleNewsLinks();
      break;
    case 'lobsters':
      links = findLobstersLinks();
      break;
    default:
      links = new Map();
  }

  console.log('[Boringfy] Total external links found:', links.size);
  return links;
}

// Replace title text in a link element
function replaceTitle(element, boringTitle) {
  if (!element.dataset.originalTitle) {
    element.dataset.originalTitle = element.textContent;
  }
  console.log('[Boringfy] Replacing:', element.textContent, '->', boringTitle);
  console.log('[Boringfy] Element tag:', element.tagName, 'class:', element.className);
  console.log('[Boringfy] Element innerHTML before:', element.innerHTML.substring(0, 200));
  element.classList.add('boringfy-replaced');
  element.textContent = boringTitle;
  console.log('[Boringfy] Element innerHTML after:', element.innerHTML.substring(0, 200));
}

// Send URLs to API and process response
async function processUrls(urlElementMap) {
  const urls = Array.from(urlElementMap.keys());
  if (urls.length === 0) return;

  console.log('[Boringfy] Sending', urls.length, 'URLs to API:', urls);

  try {
    const response = await fetch(API_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ urls }),
    });

    console.log('[Boringfy] API response status:', response.status);

    if (!response.ok) {
      console.error('[Boringfy] API error:', response.status);
      return;
    }

    const data = await response.json();
    console.log('[Boringfy] API response:', data);

    console.log('[Boringfy] URL map keys:', Array.from(urlElementMap.keys()));
    console.log('[Boringfy] API response URLs:', Object.keys(data.articles));
    
    for (const [url, info] of Object.entries(data.articles)) {
      const element = urlElementMap.get(url);
      console.log('[Boringfy] Looking up URL:', url, 'found element:', !!element);
      if (!element) {
        // Try to find a matching URL (debug)
        for (const key of urlElementMap.keys()) {
          if (key.includes(url) || url.includes(key)) {
            console.log('[Boringfy] Possible match - map has:', key);
          }
        }
        continue;
      }

      console.log('[Boringfy] Processing URL:', url, 'status:', info.status, 'boring_title:', info.boring_title);

      if (info.status === 'done' && info.boring_title) {
        replaceTitle(element, info.boring_title);
        setCachedTitle(url, info.boring_title);
        pendingUrls.delete(url);
      } else if (info.status === 'pending' || info.status === 'processing') {
        if (!pendingUrls.has(url)) {
          console.log('[Boringfy] Adding to poll queue:', url);
          pendingUrls.set(url, { pollCount: 0, element });
        }
      }
    }
  } catch (error) {
    console.error('[Boringfy] Fetch error:', error);
  }
}

// Poll for pending URLs
async function pollPending() {
  if (pendingUrls.size === 0) return;

  console.log('[Boringfy] Polling for', pendingUrls.size, 'pending URLs');

  const urlsToCheck = new Map();

  for (const [url, state] of pendingUrls.entries()) {
    state.pollCount++;
    if (state.pollCount > MAX_POLLS) {
      console.log('[Boringfy] Giving up on URL after max polls:', url);
      pendingUrls.delete(url);
    } else {
      urlsToCheck.set(url, state.element);
    }
  }

  if (urlsToCheck.size > 0) {
    await processUrls(urlsToCheck);
  }

  if (pendingUrls.size > 0) {
    setTimeout(pollPending, POLL_INTERVAL);
  }
}

// Main function - run on page load and mutations
async function boringfy() {
  if (!isEnabled) {
    console.log('[Boringfy] Skipping - extension disabled');
    return;
  }

  console.log('[Boringfy] Running boringfy()');
  const links = findArticleLinks();

  const newLinks = new Map();
  for (const [rawUrl, element] of links) {
    // Normalize URL to strip tracking params for consistent matching
    const url = normalizeUrl(rawUrl);
    
    if (!element.classList.contains('boringfy-replaced') && 
        !element.classList.contains('boringfy-pending')) {
      const cachedTitle = getCachedTitle(url);
      if (cachedTitle) {
        console.log('[Boringfy] Using cached title for:', url);
        replaceTitle(element, cachedTitle);
      } else {
        element.classList.add('boringfy-pending');
        newLinks.set(url, element);
      }
    }
  }

  console.log('[Boringfy] New links to process:', newLinks.size);

  if (newLinks.size > 0) {
    await processUrls(newLinks);

    if (pendingUrls.size > 0) {
      setTimeout(pollPending, POLL_INTERVAL);
    }
  }
}

// Watch for dynamic content (infinite scroll, etc.)
const observer = new MutationObserver((mutations) => {
  if (isEnabled) {
    clearTimeout(observer.timeout);
    observer.timeout = setTimeout(boringfy, 500);
  }
});

observer.observe(document.body, {
  childList: true,
  subtree: true,
});

console.log('[Boringfy] MutationObserver set up');
