// Browser API polyfill for cross-browser compatibility
// Firefox uses `browser.*` with Promises, Chrome uses `chrome.*` with callbacks
// This normalizes to the Promise-based `browser.*` API

if (typeof browser === 'undefined') {
  // We're in Chrome - create a browser object that wraps chrome APIs
  window.browser = {
    storage: {
      local: {
        get: (keys) => new Promise((resolve) => {
          chrome.storage.local.get(keys, resolve);
        }),
        set: (items) => new Promise((resolve) => {
          chrome.storage.local.set(items, resolve);
        })
      }
    },
    runtime: {
      onMessage: {
        addListener: (callback) => {
          chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            const result = callback(message, sender);
            if (result instanceof Promise) {
              result.then(sendResponse);
              return true; // Keep channel open for async response
            }
            return false;
          });
        }
      },
      sendMessage: (message) => new Promise((resolve) => {
        chrome.runtime.sendMessage(message, resolve);
      })
    },
    tabs: {
      query: (queryInfo) => new Promise((resolve) => {
        chrome.tabs.query(queryInfo, resolve);
      }),
      sendMessage: (tabId, message) => new Promise((resolve, reject) => {
        chrome.tabs.sendMessage(tabId, message, (response) => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            resolve(response);
          }
        });
      })
    }
  };
}
